<?php
require_once 'config.php';
require_once 'database.php';
require_once 'api_functions.php';

$user = null;
$success_message = '';
$error_message = '';

if (isset($_SESSION['email'])) {
    $stmt = $db->prepare("SELECT * FROM users WHERE email = ?");
    $stmt->execute([$_SESSION['email']]);
    $user = $stmt->fetch();
}

$page = $_GET['page'] ?? 'dashboard';

// Handle Actions (POST requests)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $user) {
    if (!$user['refresh_token'] || !$user['account_id'] || !$user['location_id']) {
        $error_message = "Your account is not fully connected.";
    } else {
        $access_token = refreshAccessToken($user['refresh_token']);
        if (!$access_token) {
            $error_message = "Failed to refresh token. Please login again.";
        } else {
            // Action Router
            $action = $_POST['action'] ?? '';

            if ($action === 'create_post') {
                $content = trim($_POST['content']);
                $post_type = $_POST['post_type'] ?? 'STANDARD';
                $schedule_time = $_POST['schedule_time'] ?? '';
                $action_url = trim($_POST['action_url']);

                // Handle Image Upload
                $media_url = null;
                if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
                    $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                    $new_name = uniqid() . '.' . $ext;
                    move_uploaded_file($_FILES['image']['tmp_name'], UPLOAD_DIR . $new_name);
                    $media_url = rtrim(DOMAIN, '/') . '/uploads/' . $new_name;
                }

                if (!empty($schedule_time)) {
                    // Save to Content Calendar (DB)
                    $stmt = $db->prepare("INSERT INTO scheduled_posts (user_email, post_type, content, media_url, schedule_time, status) VALUES (?, ?, ?, ?, ?, 'pending')");
                    $stmt->execute([$user['email'], $post_type, $content, $media_url, $schedule_time]);
                    $success_message = "Post scheduled for " . htmlspecialchars($schedule_time) . "!";
                } else {
                    // Post Immediately
                    $post_data = ["topicType" => $post_type, "summary" => $content];

                    if ($post_type === 'EVENT') {
                        $post_data['event'] = [
                            "title" => "Special Event",
                            "schedule" => [
                                "startDate" => ["year" => date('Y'), "month" => date('m'), "day" => date('d')],
                                "startTime" => ["hours" => date('H'), "minutes" => date('i')],
                                "endDate" => ["year" => date('Y'), "month" => date('m'), "day" => date('d') + 1],
                                "endTime" => ["hours" => 23, "minutes" => 59]
                            ]
                        ];
                    }

                    if (!empty($media_url)) {
                        $post_data["media"] = [
                            ["mediaFormat" => "PHOTO", "sourceUrl" => $media_url]
                        ];
                    }
                    if (!empty($action_url)) {
                        $post_data["callToAction"] = [
                            "actionType" => "LEARN_MORE",
                            "url" => $action_url
                        ];
                    }

                    $res = postToGMB($access_token, $user['account_id'], $user['location_id'], $post_data);
                    if ($res['status']) {
                        $success_message = "Post published instantly!";
                    } else {
                        $error_message = "Failed to post to GMB: " . htmlspecialchars($res['response']);
                    }
                }
            } elseif ($action === 'reply_review') {
                $review_id = $_POST['review_id'];
                $reply_text = trim($_POST['reply_text']);
                if (empty($reply_text)) {
                    $error_message = "Reply cannot be empty.";
                } else {
                    $res = replyToReview($access_token, $user['account_id'], $user['location_id'], $review_id, $reply_text);
                    if (isset($res['comment'])) {
                        $success_message = "Reply sent successfully!";
                    } else {
                        $error_message = "Failed to reply.";
                    }
                }
            } elseif ($action === 'select_location') {
                $location_id = trim($_POST['location_id']); // actually the short ID
                $account_id = trim($_POST['account_id']);
                $location_name = trim($_POST['location_name']);
                if ($location_id && $account_id) {
                    $stmt = $db->prepare("UPDATE users SET account_id = ?, location_id = ?, location_name = ? WHERE email = ?");
                    $stmt->execute([$account_id, $location_id, $location_name, $user['email']]);
                    // Refresh user data
                    $user['account_id'] = $account_id;
                    $user['location_id'] = $location_id;
                    $user['location_name'] = $location_name;
                    $success_message = "Successfully connected to the new location: " . htmlspecialchars($location_name);
                } else {
                    $error_message = "Invalid location selected.";
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - GMB SAAS on onnac.com</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src='https://cdn.jsdelivr.net/npm/fullcalendar@6.1.15/index.global.min.js'></script>
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f8fafc;
        }
    </style>
</head>

<body class="text-gray-800">

    <?php if (!$user): ?>
        <div class="flex items-center justify-center min-h-screen text-center">
            <div class="bg-white p-10 rounded-2xl shadow-xl max-w-sm w-full border border-gray-100">
                <h1 class="text-2xl font-bold mb-2">GMB Auto-Poster</h1>
                <p class="text-gray-500 mb-8 text-sm">Automate Google My Business Posts, Reviews & Insights on Onnac.com</p>
                <a href="login.php"
                    class="flex justify-center gap-3 px-6 py-3 border border-gray-300 rounded-lg shadow-sm bg-white hover:bg-gray-50 font-medium">
                    <img src="https://upload.wikimedia.org/wikipedia/commons/c/c1/Google_%22G%22_logo.svg" alt="Google"
                        class="w-5 h-5"> Login with Google
                </a>
            </div>
        </div>
    <?php else: ?>

        <!-- Sidebar Layout -->
        <div class="flex h-screen overflow-hidden">

            <!-- Sidebar -->
            <aside class="w-64 bg-white border-r border-gray-200 flex-shrink-0 flex flex-col hidden sm:flex">
                <div class="h-16 flex items-center px-6 border-b border-gray-200">
                    <span class="text-xl font-bold text-gray-900">Onnac GMB</span>
                </div>
                <nav class="flex-1 px-4 py-6 space-y-2 overflow-y-auto">
                    <a href="?page=dashboard"
                        class="<?= $page === 'dashboard' ? 'bg-blue-50 text-blue-700' : 'text-gray-600 hover:bg-gray-50' ?> flex items-center px-3 py-2.5 rounded-lg font-medium transition">
                        Dashboard & Insights
                    </a>
                    <a href="?page=posts"
                        class="<?= $page === 'posts' ? 'bg-blue-50 text-blue-700' : 'text-gray-600 hover:bg-gray-50' ?> flex items-center px-3 py-2.5 rounded-lg font-medium transition">
                        Create Posts & Media
                    </a>
                    <a href="?page=calendar"
                        class="<?= $page === 'calendar' ? 'bg-blue-50 text-blue-700' : 'text-gray-600 hover:bg-gray-50' ?> flex items-center px-3 py-2.5 rounded-lg font-medium transition">
                        Content Calendar
                    </a>
                    <a href="?page=reviews"
                        class="<?= $page === 'reviews' ? 'bg-blue-50 text-blue-700' : 'text-gray-600 hover:bg-gray-50' ?> flex items-center px-3 py-2.5 rounded-lg font-medium transition">
                        Manage Reviews
                    </a>
                    <a href="?page=settings"
                        class="<?= $page === 'settings' ? 'bg-blue-50 text-blue-700' : 'text-gray-600 hover:bg-gray-50' ?> flex items-center px-3 py-2.5 rounded-lg font-medium transition">
                        Connection Setup
                    </a>
                </nav>
                <div class="p-4 border-t border-gray-200 text-sm">
                    <p class="text-gray-600 truncate"><?= htmlspecialchars($user['email']) ?></p>
                    <a href="logout.php" class="text-red-500 hover:text-red-600 font-medium mt-1 block">Logout</a>
                </div>
            </aside>

            <!-- Main Content -->
            <main class="flex-1 flex flex-col h-screen overflow-hidden">
                <header class="h-16 bg-white border-b border-gray-200 flex items-center px-6 sm:px-8 shadow-sm">
                    <h2 class="text-xl font-semibold capitalize"><?= str_replace('_', ' ', $page) ?></h2>
                </header>

                <div class="flex-1 overflow-y-auto p-6 sm:p-8">

                    <?php if ($success_message): ?>
                        <div class="mb-6 p-4 rounded-xl bg-green-50 border border-green-200 text-green-800 text-sm font-medium">
                            <?= htmlspecialchars($success_message) ?>
                        </div>
                    <?php endif; ?>
                    <?php if ($error_message): ?>
                        <div class="mb-6 p-4 rounded-xl bg-red-50 border border-red-200 text-red-800 text-sm font-medium">
                            <?= htmlspecialchars($error_message) ?>
                        </div>
                    <?php endif; ?>

                    <?php
                    // ROUTING PAGES
                    if ($page === 'dashboard'):
                        ?>
                        <!-- Insights & Performance -->
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
                            <div class="bg-white p-6 rounded-2xl shadow-sm border border-gray-200">
                                <h3 class="text-sm font-semibold text-gray-500 mb-2">Google Client Connection</h3>
                                <?php if (!empty($user['location_id'])): ?>
                                    <p class="text-lg font-bold text-green-600">Active Location</p>
                                                <p class="text-sm text-gray-800 font-semibold mt-1"><?= htmlspecialchars($user['location_name'] ?? 'Unknown Location') ?></p>
                                    <p class="text-xs text-gray-400 mt-1">Loc ID: <?= htmlspecialchars($user['location_id']) ?></p>
                                <?php else: ?>
                                    <p class="text-lg font-bold text-red-600">Disconnected</p>
                                    <p class="text-sm text-red-500 mt-1">No setup found.</p>
                                    <?php if (!empty($_SESSION['gmb_debug'])): ?>
                                        <div class="mt-3 p-3 bg-red-50 border border-red-200 text-xs text-red-700 rounded break-words">
                                            <strong>Debug Info:</strong> <?= htmlspecialchars($_SESSION['gmb_debug']) ?>
                                        </div>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </div>
                            <div class="bg-white p-6 rounded-2xl shadow-sm border border-gray-200">
                                <h3 class="text-sm font-semibold text-gray-500 mb-2">Posts Scheduled</h3>
                                <?php
                                $stmt = $db->prepare("SELECT COUNT(*) FROM scheduled_posts WHERE user_email = ? AND status = 'pending'");
                                $stmt->execute([$user['email']]);
                                $count = $stmt->fetchColumn();
                                ?>
                                <p class="text-3xl font-extrabold text-blue-600"><?= $count ?></p>
                            </div>
                        </div>

                        <div class="bg-white p-6 rounded-2xl shadow-sm border border-gray-200 mb-8">
                            <h3 class="text-lg font-bold text-gray-800 mb-4">Location Performance (Mockup / Real Data via
                                Insights API requires approval)</h3>
                            <p class="text-gray-500 text-sm mb-4">Google Business APIs demand elevated permissions to fetch
                                Insights dynamically. Ye section properly kaam karega jab aap 'mybusiness.googleapis.com' ki
                                APIs approve karwalenge.</p>
                            <canvas id="performanceChart" height="100"></canvas>
                            <script>
                                const ctx = document.getElementById('performanceChart').getContext('2d');
                                new Chart(ctx, {
                                    type: 'line',
                                    data: {
                                        labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                                        datasets: [{
                                            label: 'Profile Views',
                                            data: [12, 19, 3, 5, 2, 3, 9],
                                            borderColor: '#2563eb',
                                            tension: 0.4
                                        }]
                                    }
                                });
                            </script>
                        </div>

                    <?php elseif ($page === 'posts'): ?>

                        <!-- Create Post Form Structure -->
                        <div class="bg-white rounded-2xl shadow-sm border border-gray-200">
                            <div class="p-6 border-b border-gray-100">
                                <h3 class="text-lg font-bold">Create New Post</h3>
                            </div>
                            <form method="POST" enctype="multipart/form-data" class="p-6 space-y-6">
                                <input type="hidden" name="action" value="create_post">

                                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Post Type</label>
                                        <select name="post_type"
                                            class="w-full border-gray-300 rounded-lg p-3 border outline-none focus:ring-2 focus:ring-blue-500">
                                            <option value="STANDARD">Standard Update</option>
                                            <option value="EVENT">Event</option>
                                            <option value="OFFER">Offer</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Button URL (Action)</label>
                                        <input type="url" name="action_url" placeholder="https://example.com"
                                            class="w-full border-gray-300 rounded-lg p-3 border outline-none focus:ring-2 focus:ring-blue-500">
                                    </div>
                                </div>

                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Upload Image (Optional)</label>
                                    <input type="file" name="image" accept="image/png, image/jpeg"
                                        class="w-full border-gray-300 rounded-lg p-2 border outline-none focus:ring-2 focus:ring-blue-500 bg-gray-50">
                                    <p class="text-xs text-gray-400 mt-1">Image must be local & accessible over the internet via
                                        domain link during upload.</p>
                                </div>

                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Post Content / Details</label>
                                    <textarea name="content" rows="4" required
                                        class="w-full border-gray-300 rounded-lg p-3 border outline-none focus:ring-2 focus:ring-blue-500 resize-none"
                                        placeholder="Write something exciting..."></textarea>
                                </div>

                                <div class="bg-gray-50 p-4 rounded-xl border border-gray-200">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Schedule Time (Leave blank to
                                        post now)</label>
                                    <input type="datetime-local" name="schedule_time"
                                        class="w-full md:w-1/2 border-gray-300 rounded-lg p-2 border outline-none focus:ring-2 focus:ring-blue-500">
                                </div>

                                <button type="submit"
                                    class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-3 px-6 rounded-xl transition shadow">
                                    Publish or Schedule
                                </button>
                            </form>
                        </div>

                    <?php elseif ($page === 'calendar'): ?>

                        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                            <div class="lg:col-span-2 bg-white rounded-2xl shadow-sm border border-gray-200">
                                <div class="p-6 border-b border-gray-100">
                                    <h3 class="text-lg font-bold">Post Calendar</h3>
                                    <p class="text-sm text-gray-500">Upcoming and past posts visually displayed</p>
                                </div>
                                <div class="p-6">
                                    <div id='calendar'></div>
                                    <?php
                                    $stmt = $db->prepare("SELECT * FROM scheduled_posts WHERE user_email = ?");
                                    $stmt->execute([$user['email']]);
                                    $schedules = $stmt->fetchAll();
                                    
                                    $calendar_events = [];
                                    foreach ($schedules as $s) {
                                        $color = $s['status'] === 'pending' ? '#3b82f6' : '#22c55e'; // blue or green
                                        $title = "[". $s['post_type'] . "] " . substr($s['content'], 0, 20) . "...";
                                        $calendar_events[] = [
                                            'title' => $title,
                                            'start' => $s['schedule_time'],
                                            'color' => $color
                                        ];
                                    }
                                    ?>
                                    <script>
                                      document.addEventListener('DOMContentLoaded', function() {
                                        var calendarEl = document.getElementById('calendar');
                                        var calendar = new FullCalendar.Calendar(calendarEl, {
                                          initialView: 'dayGridMonth',
                                          height: 500,
                                          events: <?= json_encode($calendar_events) ?>,
                                          eventClick: function(info) {
                                              alert('Post Details: ' + info.event.title);
                                          }
                                        });
                                        calendar.render();
                                      });
                                    </script>
                                </div>
                            </div>
                            
                            <div class="bg-white rounded-2xl shadow-sm border border-gray-200">
                                <div class="p-6 border-b border-gray-100">
                                    <h3 class="text-md font-bold">List View (Upcoming)</h3>
                                </div>
                                <div class="p-6">
                                    <ul class="space-y-4">
                                        <?php
                                        $upcoming = array_filter($schedules, fn($s) => $s['status'] === 'pending');
                                        usort($upcoming, fn($a, $b) => strtotime($a['schedule_time']) - strtotime($b['schedule_time']));
                                        $upcoming = array_slice($upcoming, 0, 5); // show next 5
                                        foreach ($upcoming as $u):
                                        ?>
                                        <li class="border-l-4 border-blue-500 pl-4 py-1">
                                            <p class="text-xs text-gray-400 font-bold mb-1"><?= date('M d, g:i A', strtotime($u['schedule_time'])) ?></p>
                                            <p class="text-sm text-gray-800 font-medium whitespace-nowrap overflow-hidden text-ellipsis"><?= htmlspecialchars($u['content']) ?></p>
                                        </li>
                                        <?php endforeach;
                                        if (empty($upcoming)) {
                                            echo '<p class="text-sm text-gray-500">No upcoming posts scheduled.</p>';
                                        }
                                        ?>
                                    </ul>
                                </div>
                            </div>
                        </div>

                    <?php elseif ($page === 'reviews'): ?>

                        <div class="bg-white rounded-2xl shadow-sm border border-gray-200">
                            <div class="p-6 border-b border-gray-100">
                                <h3 class="text-lg font-bold">Recent Reviews (Mock/Logic)</h3>
                            </div>
                            <div class="p-6 space-y-6">
                                <?php
                                // This gets real reviews if API gives access, else mocks for display.
                                $access_token = refreshAccessToken($user['refresh_token']);
                                if ($access_token && !empty($user['location_id'])) {
                                    $reviewsApi = fetchGmbReviews($access_token, $user['account_id'], $user['location_id']);
                                    $reviews = $reviewsApi['reviews'] ?? []; // Fallback to empty if not permitted
                                } else {
                                    $reviews = [];
                                }

                                if (empty($reviews)) {
                                    echo '<div class="text-gray-400 text-center py-6">No real reviews found or elevated API access missing for reading reviews. Here is strictly the layout that will populate.</div>';
                                    // Mock data to show UI
                                    $reviews = [
                                        ['reviewId' => '1', 'reviewer' => ['displayName' => 'John Doe'], 'starRating' => 'FIVE', 'comment' => 'Great Business! Highly recommend their products.', 'createTime' => '2026-02-23T10:00:00Z']
                                    ];
                                }

                                foreach ($reviews as $rev):
                                    ?>
                                    <div class="border rounded-xl p-5 border-gray-200">
                                        <div class="flex items-center gap-3 mb-2">
                                            <div
                                                class="bg-gray-200 rounded-full w-10 h-10 flex items-center justify-center font-bold text-gray-500">
                                                <?= substr($rev['reviewer']['displayName'], 0, 1) ?>
                                            </div>
                                            <div>
                                                <h4 class="font-bold text-sm text-gray-900">
                                                    <?= htmlspecialchars($rev['reviewer']['displayName']) ?>
                                                </h4>
                                                <p class="text-xs text-yellow-500 font-bold">
                                                    <?= htmlspecialchars($rev['starRating']) ?> STARS
                                                </p>
                                            </div>
                                        </div>
                                        <p class="text-sm text-gray-700 italic mb-4">
                                            "<?= htmlspecialchars($rev['comment'] ?? 'No comment provided.') ?>"</p>

                                        <!-- Reply Form -->
                                        <form method="POST" class="mt-2 text-sm flex gap-3">
                                            <input type="hidden" name="action" value="reply_review">
                                            <input type="hidden" name="review_id" value="<?= htmlspecialchars($rev['reviewId']) ?>">
                                            <input type="text" name="reply_text" placeholder="Write a reply..."
                                                class="flex-1 border-gray-300 rounded p-2 border outline-none focus:border-blue-500">
                                            <button type="submit"
                                                class="bg-gray-800 text-white px-4 py-2 rounded hover:bg-black transition">Reply</button>
                                        </form>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>

                    <?php elseif ($page === 'settings'): ?>

                        <div class="bg-white rounded-2xl shadow-sm border border-gray-200">
                            <div class="p-6 border-b border-gray-100">
                                <h3 class="text-lg font-bold">Available Google Accounts & Locations</h3>
                            </div>
                            <div class="p-6 space-y-6">
                                <p class="text-sm text-gray-500">
                                    Neechay aapke sub accounts aur unme majood GMB locations ki detail aayegi. Yahan se apna
                                    desired business choose karein.
                                </p>
                                <?php
                                $access_token = refreshAccessToken($user['refresh_token']);
                                if (!$access_token):
                                    echo '<div class="text-red-500 text-sm">Token refresh fail ho gaya. Kripya naye sire se login karein.</div>';
                                else:
                                    $accountsApi = getGmbAccounts($access_token);
                                    if (isset($accountsApi['error'])) {
                                        echo '<div class="bg-red-50 text-red-700 p-4 rounded text-sm break-all font-mono"><strong>Accounts API Error:</strong><br>' . htmlspecialchars(json_encode($accountsApi['error'])) . '</div>';
                                    } elseif (empty($accountsApi['accounts'])) {
                                        echo '<div class="p-4 border rounded text-gray-500 text-sm">Koi GMB Account nahi mila is login mein.</div>';
                                    } else {
                                        foreach ($accountsApi['accounts'] as $acc) {
                                            $acc_id_short = explode('/', $acc['name'])[1];
                                            echo '<div class="mb-6 border border-gray-200 rounded-xl overflow-hidden">';
                                            echo '<div class="bg-gray-50 p-4 border-b border-gray-200 font-bold">' . htmlspecialchars($acc['accountName']) . ' <span class="text-xs text-gray-400 font-normal">(' . $acc_id_short . ')</span></div>';

                                            // Fetch Locations for this account
                                            $locationsApi = getGmbLocations($access_token, $acc_id_short);

                                            if (isset($locationsApi['error'])) {
                                                echo '<div class="p-4 text-red-600 text-xs font-mono"><strong>Locations API Error:</strong><br>' . htmlspecialchars(json_encode($locationsApi['error'])) . '</div>';
                                            } elseif (empty($locationsApi['locations'])) {
                                                echo '<div class="p-4 text-sm text-gray-500">Is account main koi location nahi ha.</div>';
                                            } else {
                                                echo '<ul class="divide-y divide-gray-100">';
                                                foreach ($locationsApi['locations'] as $loc) {
                                                    $loc_id_short = explode('/', $loc['name'])[3];
                                                    $loc_title = $loc['title'] ?? 'Unknown Location';
                                                    $is_active = ($user['location_id'] === $loc_id_short);
                                                    echo '<li class="p-4 flex items-center justify-between hover:bg-gray-50 transition">';
                                                    echo '<div>';
                                                    echo '<h4 class="font-bold text-sm text-gray-900">' . htmlspecialchars($loc_title) . '</h4>';
                                                    echo '<p class="text-xs text-gray-500">ID: ' . $loc_id_short . '</p>';
                                                    echo '</div>';

                                                    if ($is_active) {
                                                        echo '<span class="bg-green-100 text-green-700 px-3 py-1 rounded-full text-xs font-bold shadow-sm border border-green-200">Currently Active</span>';
                                                    } else {
                                                        echo '<form method="POST">';
                                                        echo '<input type="hidden" name="action" value="select_location">';
                                                        echo '<input type="hidden" name="account_id" value="' . htmlspecialchars($acc_id_short) . '">';
                                                        echo '<input type="hidden" name="location_id" value="' . htmlspecialchars($loc_id_short) . '">';
                                                        echo '<input type="hidden" name="location_name" value="' . htmlspecialchars($loc_title) . '">';
                                                        echo '<button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-1.5 rounded-lg text-sm font-semibold transition">Connect</button>';
                                                        echo '</form>';
                                                    }

                                                    echo '</li>';
                                                }
                                                echo '</ul>';
                                            }
                                            echo '</div>';
                                        }
                                    }
                                endif;
                                ?>
                            </div>
                        </div>

                    <?php endif; ?>

                </div>
            </main>
        </div>
    <?php endif; ?>

</body>

</html>